"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var react_1 = require("react");
var isClient_1 = __importDefault(require("./shared/isClient"));
var isAPISupported_1 = __importDefault(require("./shared/isAPISupported"));
var warnOnce_1 = __importDefault(require("./shared/warnOnce"));
var errorMessage = 'matchMedia is not supported, this could happen both because window.matchMedia is not supported by' +
    ' your current browser or you\'re using the useMediaQuery hook whilst server side rendering.';
/**
 * Accepts a media query string then uses the
 * [window.matchMedia](https://developer.mozilla.org/en-US/docs/Web/API/Window/matchMedia) API to determine if it
 * matches with the current document.<br />
 * It also monitor the document changes to detect when it matches or stops matching the media query.<br />
 * Returns the validity state of the given media query.
 *
 */
var useMediaQuery = function (mediaQuery) {
    if (!isClient_1.default || !(0, isAPISupported_1.default)('matchMedia')) {
        (0, warnOnce_1.default)(errorMessage);
        return false;
    }
    var _a = (0, react_1.useState)(!!window.matchMedia(mediaQuery).matches), isVerified = _a[0], setIsVerified = _a[1];
    (0, react_1.useEffect)(function () {
        var mediaQueryList = window.matchMedia(mediaQuery);
        var documentChangeHandler = function () { setIsVerified(!!mediaQueryList.matches); };
        try {
            mediaQueryList.addEventListener('change', documentChangeHandler);
        }
        catch (e) {
            // Safari isn't supporting mediaQueryList.addEventListener
            mediaQueryList.addListener(documentChangeHandler);
        }
        documentChangeHandler();
        return function () {
            try {
                mediaQueryList.removeEventListener('change', documentChangeHandler);
            }
            catch (e) {
                // Safari isn't supporting mediaQueryList.removeEventListener
                mediaQueryList.removeListener(documentChangeHandler);
            }
        };
    }, [mediaQuery]);
    return isVerified;
};
exports.default = useMediaQuery;
